<?php
/**
 * Class to handle Instructor Emails
 * @since   3.2.1
 */
if (! class_exists('irInstructorEmails')) {
    class irInstructorEmails
    {
        protected $plugin_slug = '';
        protected static $instance = null;

        public function __construct()
        {
            $this->plugin_slug = 'wdm_instructor_role';

            add_filter('ir_filter_instructor_tabs', array($this, 'ir_add_instructor_email_tab'), 10, 1);
            add_action('instuctor_tab_checking', array($this, 'ir_add_instructor_email_tab_content'), 10, 1);
            add_action('woocommerce_order_status_completed', array($this, 'ir_send_course_purchase_email_to_instructor'), 10, 1);
        }

        public static function getInstance()
        {
            if (null == self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }

        /**
         * Add Instructor emails tab
         *
         * @since   3.2.1
         */
        public function ir_add_instructor_email_tab($tabs)
        {
            if (function_exists('wdm_is_instructor') && wdm_is_instructor()) {
                // Check if instructor emails setting enabled.
                $ir_settings = get_option('_wdmir_admin_settings', array());
                if (! array_key_exists('instructor_mail', $ir_settings) || $ir_settings['instructor_mail'] != 1) {
                    return $tabs;
                }
                $tabs['instructor-email'] = __('Instructor Emails', 'wdm_instructor_role');
            }
            return $tabs;
        }

        /**
         * Add instructor email tab content
         *
         * @since   3.2.1
         */
        public function ir_add_instructor_email_tab_content($current_tab)
        {
            if (! function_exists('wdm_is_instructor') && ! wdm_is_instructor()) {
                return;
            }
            $user_id = get_current_user_id();
            if ('instructor-email' === $current_tab) {
                if (array_key_exists('instructor-email-save', $_POST) && ! empty($_POST['instructor-email-save'])) {
                    $this->ir_save_instructor_email_settings($user_id);
                }
                $subject = get_user_meta($user_id, 'ir-course-purchase-email-sub', 1);
                $body = get_user_meta($user_id, 'ir-course-purchase-email-body', 1);
                ?>
                <h2><?php esc_html(_e('Course Purchase Email', 'wdm_instructor_role'))?></h2>
                <p>
                    <strong><?php esc_html(_e('Description : ', 'wdm_instructor_role'))?></strong>
                    <?php esc_html_e('Now receive a email notification whenever your course is purchased through a product. You can customize the contents of that email from here.', 'wdm_instructor_role'); ?>
                </p>
                <form method="post">
                    <p>
                        <strong><?php esc_html(_e('Available shortcodes : ', 'wdm_instructor_role'))?></strong>
                        <?php _e('<em>[site_name]</em>, <em>[course_name]</em>, <em>[instructor_name]</em> and <em>[customer_name]</em>', 'wdm_instructor_role')?>
                    </p>
                    <table class="ir-course-purchase-email-body">
                        <tbody>
                            <tr scope="row">
                                <th class="ir-email-settings-label">
                                    <?php esc_html(_e('Subject', 'wdm_instructor_role'))?>
                                </th>
                                <td>
                                    <input type="text" name="ir-course-purchase-email-sub" value="<?php echo $subject; ?>">
                                </td>
                            </tr>
                            <tr scope="row">
                                <th class="ir-email-settings-label">
                                    <?php esc_html(_e('Body', 'wdm_instructor_role'))?>
                                </th>
                                <td>
                                    <?php wp_editor(
                                        $body,
                                        'ir-course-purchase-email-body',
                                        array('media_buttons' => false)
                                    ); ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                    <input type="submit" class="button-primary" name="instructor-email-save" value="<?php esc_html(_e('Save', 'wdm_instructor_role')); ?>" />
                </form>
                <?php
            }
        }

        /**
         * Save instructor email settings
         *
         * @since   3.2.1
         */
        public function ir_save_instructor_email_settings($user_id)
        {
            if (empty($user_id)) {
                $user_id = get_current_user_id();
            }

            $subject = filter_input(INPUT_POST, 'ir-course-purchase-email-sub', FILTER_SANITIZE_STRING);
            $body = wpautop($_POST['ir-course-purchase-email-body']);

            update_user_meta($user_id, 'ir-course-purchase-email-sub', $subject);
            update_user_meta($user_id, 'ir-course-purchase-email-body', $body);
        }

        /**
         * Send course purchase emails to instructors
         *
         * @since   3.2.1
         */
        public function ir_send_course_purchase_email_to_instructor($order_id)
        {
            $order = new WC_Order($order_id);
            $items = $order->get_items();
            
            foreach ($items as $item) {
                $product_id = $item['product_id'];
                $total = $item['line_total'];
                $product_post = get_post($product_id);
                $author_id = $product_post->post_author;
                $related_course = get_post_meta($product_id,'_related_course',true);
                $course_id = $related_course[0];
                $assigned_course = get_post($course_id);

            if (! wdm_is_instructor($author_id)) {
                if (! empty($related_course)) {
                    $author_id = $assigned_course->post_author;
                }
            }

                if (wdm_is_instructor($author_id)) {
                    $subject = get_user_meta($author_id, 'ir-course-purchase-email-sub', 1);
                    $body = get_user_meta($author_id, 'ir-course-purchase-email-body', 1);

                    if (empty($subject)) {
                        $subject = sprintf(__('[ %s ] : Course Purchase Email', 'wdm_is_instructor'), get_bloginfo('name'));
                    }

                    if (empty($body)) {
                        $body = '<p>'.sprintf(__('Hello %s ,', 'wdm_is_instructor'), get_user_meta($author_id, 'first_name', 1).' '.get_user_meta($author_id, 'last_name', 1), 1).'</p>';
                        $body .= '<p>'.sprintf(__('A new purchase has been made for your course <strong>%s</strong> by <strong>%s</strong>.', 'wdm_is_instructor'), $assigned_course->post_title, $order->get_billing_first_name().' '.$order->get_billing_last_name()).'</p>';
                        $body .= '<p>'.__('Thank You', 'wdm_is_instructor').'</p>';
                    }

                    $find = array(
                        '[site_name]',
                        '[course_name]',
                        '[customer_name]',
                        '[instructor_name]',
                    );
                    $replace = array(
                        get_bloginfo('name'),
                        $assigned_course->post_title,
                        $order->get_billing_first_name().' '.$order->get_billing_last_name(),
                        get_user_meta($author_id, 'first_name', 1).' '.get_user_meta($author_id, 'last_name', 1)
                    );

                    $subject = str_replace($find, $replace, $subject);
                    $body = str_replace($find, $replace, $body);
                    $author_data = get_userdata($author_id);

                    if (! wp_mail($author_data->user_email, $subject, $body, array('Content-Type: text/html; charset=UTF-8'))) {
                        error_log("For Order : $order_id :: Instructor Email not sent successfully");
                    }
                }
            }
        }
    }
}