<?php
/**
 * Plugin Name: LearnDash - ConvertKit
 * Plugin URI: https://realbigplugins.com/plugins/learndash-convertkit
 * Description: ConvertKit Integration for LearnDash
 * Version: 1.1.0
 * Text Domain: learndash-convertkit
 * Author: Real Big Plugins
 * Author URI: https://realbigplugins.com
 * Contributors: joelworsham
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'LearnDash_ConvertKit' ) ) {

	define( 'LEARNDASH_CONVERTKIT_VER', '1.1.0' );
	define( 'LEARNDASH_CONVERTKIT_DIR', plugin_dir_path( __FILE__ ) );
	define( 'LEARNDASH_CONVERTKIT_URL', plugin_dir_url( __FILE__ ) );
	define( 'LEARNDASH_CONVERTKIT_FILE', __FILE__ );

	/**
	 * Class LearnDash_ConvertKit
	 *
	 * Main plugin class.
	 *
	 * @since 1.0.0
	 */
	final class LearnDash_ConvertKit {

		/**
		 * RBP Support module
		 *
		 * @since 1.0.0
		 *
		 * @var RBP_Support
		 */
		public $support;

		/**
		 * Field Helpers module.
		 *
		 * @since 1.0.0
		 *
		 * @var RBM_FieldHelpers
		 */
		public $field_helpers;

		/**
		 * ConvertKit Upgrade class
		 *
		 * @since 1.1.0
		 *
		 * @var LD_ConvertKit_Upgrade
		 */
		public $upgrade;

		/**
		 * ConvertKit API class
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_API
		 */
		public $convertkit_api = false;

		/**
		 * Admin.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_Admin
		 */
		public $admin;

		/**
		 * Admin Post Edit Course.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_PostEditCourse
		 */
		public $admin_posteditcourse;

		/**
		 * User profile.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_UserProfile
		 */
		public $user_profile;

		/**
		 * License field.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_License_Field
		 */
		public $license_field;

		/**
		 * Courses integration.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_Courses
		 */
		public $courses;

		/**
		 * Resource Tagging integration.
		 *
		 * @since 1.1.0
		 *
		 * @var LD_ConvertKit_ResourceTagging
		 */
		public $resource_tagging;

		/**
		 * Shortcodes.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_Shortcodes
		 */
		public $shortcodes;

		/**
		 * Widgets.
		 *
		 * @since 1.0.0
		 *
		 * @var LD_ConvertKit_Widgets
		 */
		public $widgets;

		protected function __wakeup() {
		}

		protected function __clone() {
		}

		/**
		 * Get active instance
		 *
		 * @since 1.0.0
		 *
		 * @return LearnDash_ConvertKit The one true LearnDash_convertkit
		 */
		public static function instance() {

			static $instance = null;

			if ( null === $instance ) {
				$instance = new static();
			}

			return $instance;
		}

		/**
		 * LearnDash_ConvertKit constructor.
		 *
		 * @since 1.0.0
		 */
		protected function __construct() {

			$this->load_textdomain();
			$this->setup_fieldhelpers();
			$this->setup_support();
			$this->setup_api();
			$this->require_necessities();

			// Register our CSS/JS for the whole plugin
			add_action( 'init', array( $this, 'register_assets' ) );
		}

		/**
		 * Internationalization.
		 *
		 * @since 1.0.0
		 * @access private
		 */
		private function load_textdomain() {

			// Set filter for language directory
			$lang_dir = LEARNDASH_CONVERTKIT_DIR . '/languages/';
			$lang_dir = apply_filters( 'learndash_convertkit_languages_directory', $lang_dir );

			// Traditional WordPress plugin locale filter
			$locale = apply_filters( 'plugin_locale', get_locale(), 'learndash-convertkit' );
			$mofile = sprintf( '%1$s-%2$s.mo', 'learndash-convertkit', $locale );

			// Setup paths to current locale file
			$mofile_local  = $lang_dir . $mofile;
			$mofile_global = WP_LANG_DIR . '/learndash-convertkit/' . $mofile;

			if ( file_exists( $mofile_global ) ) {

				// Look in global /wp-content/languages/learndash-convertkit/ folder
				// This way translations can be overridden via the Theme/Child Theme
				load_textdomain( 'learndash-convertkit', $mofile_global );

			} else if ( file_exists( $mofile_local ) ) {

				// Look in local /wp-content/plugins/learndash-convertkit/languages/ folder
				load_textdomain( 'learndash-convertkit', $mofile_local );

			} else {

				// Load the default language files
				load_plugin_textdomain( 'learndash-convertkit', false, $lang_dir );
			}
		}

		/**
		 * Requires and loads required files.
		 *
		 * @since 1.0.0
		 * @access private
		 */
		private function require_necessities() {

			require_once 'core/ld-convertkit-functions.php';
			require_once 'core/class-ld-convertkit-upgrade.php';
			require_once 'core/class-ld-convertkit-courses.php';
			require_once 'core/class-ld-convertkit-resourcetagging.php';
			require_once 'core/widgets/class-ld-convertkit-widgets.php';
			require_once 'core/shortcodes/class-ld-convertkit-shortcodes.php';

			$this->upgrade          = new LD_ConvertKit_Upgrade();
			$this->courses          = new LD_ConvertKit_Courses();
			$this->resource_tagging = new LD_ConvertKit_ResourceTagging();
			$this->shortcodes       = new LD_ConvertKit_Shortcodes();
			$this->widgets          = new LD_ConvertKit_Widgets();

			if ( is_admin() ) {

				require_once 'core/admin/class-ld-convertkit-admin.php';
				require_once 'core/admin/class-ld-convertkit-post-edit-course.php';
				require_once 'core/admin/class-ld-convertkit-user-profile.php';
				require_once 'core/admin/class-ld-convertkit-licensing-field.php';

				$this->admin                = new LD_ConvertKit_Admin();
				$this->admin_posteditcourse = new LD_ConvertKit_PostEditCourse();
				$this->user_profile         = new LD_ConvertKit_UserProfile();
				$this->license_field        = new LD_ConvertKit_License_Field();
			}
		}

		/**
		 * Initializes Field Helpers.
		 *
		 * @since @since 1.0.0
		 * @access private
		 */
		private function setup_fieldhelpers() {

			require_once 'core/library/rbm-field-helpers/rbm-field-helpers.php';
			require_once 'core/ld-convertkit-fieldhelper-functions.php';

			$this->field_helpers = new RBM_FieldHelpers( array(
				'ID'   => 'ld_convertkit',
				'l10n' => array(
					'field_table'    => array(
						'delete_row'    => __( 'Delete Row', 'learndash-convertkit' ),
						'delete_column' => __( 'Delete Column', 'learndash-convertkit' ),
					),
					'field_select'   => array(
						'no_options'       => __( 'No select options.', 'learndash-convertkit' ),
						'error_loading'    => __( 'The results could not be loaded', 'learndash-convertkit' ),
						/* translators: %d is number of characters over input limit */
						'input_too_long'   => __( 'Please delete %d character(s)', 'learndash-convertkit' ),
						/* translators: %d is number of characters under input limit */
						'input_too_short'  => __( 'Please enter %d or more characters', 'learndash-convertkit' ),
						'loading_more'     => __( 'Loading more results...', 'learndash-convertkit' ),
						/* translators: %d is maximum number items selectable */
						'maximum_selected' => __( 'You can only select %d item(s)', 'learndash-convertkit' ),
						'no_results'       => __( 'No results found', 'learndash-convertkit' ),
						'searching'        => __( 'Searching...', 'learndash-convertkit' ),
					),
					'field_repeater' => array(
						'collapsable_title' => __( 'New Row', 'learndash-convertkit' ),
						'confirm_delete'    => __( 'Are you sure you want to delete this element?', 'learndash-convertkit' ),
						'delete_item'       => __( 'Delete', 'learndash-convertkit' ),
						'add_item'          => __( 'Add', 'learndash-convertkit' ),
					),
					'field_media'    => array(
						'button_text'        => __( 'Upload / Choose Media', 'learndash-convertkit' ),
						'button_remove_text' => __( 'Remove Media', 'learndash-convertkit' ),
						'window_title'       => __( 'Choose Media', 'learndash-convertkit' ),
					),
					'field_checkbox' => array(
						'no_options_text' => __( 'No options available.', 'learndash-convertkit' ),
					),
				),
			) );
		}

		/**
		 * Initialize RBP Support.
		 *
		 * @since 1.0.0
		 * @access private
		 */
		private function setup_support() {

			require_once 'core/library/rbp-support/rbp-support.php';

			$this->support = new RBP_Support( __FILE__, array(
				'support_form'           => array(
					'enabled'  => array(
						'title'           => _x( 'Need some help with %s?', '%s is the Plugin Name', 'learndash-convertkit' ),
						'subject_label'   => __( 'Subject', 'learndash-convertkit' ),
						'message_label'   => __( 'Message', 'learndash-convertkit' ),
						'send_button'     => __( 'Send', 'learndash-convertkit' ),
						'subscribe_text'  => _x( 'We make other cool plugins and share updates and special offers to anyone who %ssubscribes here%s.', 'Both %s are used to place HTML for the <a> in the message', 'learndash-convertkit' ),
						'validationError' => _x( 'This field is required', 'Only used by legacy browsers for JavaScript Form Validation', 'learndash-convertkit' ),
						'success'         => __( 'Support message succesfully sent!', 'learndash-convertkit' ),
						'error'           => __( 'Could not send support message.', 'learndash-convertkit' ),
					),
					'disabled' => array(
						'title'            => _x( 'Need some help with %s?', '%s is the Plugin Name', 'learndash-convertkit' ),
						'disabled_message' => __( 'Premium support is disabled. Please register your product and activate your license for this website to enable.', 'learndash-convertkit' )
					),
				),
				'licensing_fields'       => array(
					'title'                    => _x( '%s License', '%s is the Plugin Name', 'learndash-convertkit' ),
					'deactivate_button'        => __( 'Deactivate', 'learndash-convertkit' ),
					'activate_button'          => __( 'Activate', 'learndash-convertkit' ),
					'delete_deactivate_button' => __( 'Delete and Deactivate', 'learndash-convertkit' ),
					'delete_button'            => __( 'Delete', 'learndash-convertkit' ),
					'license_active_label'     => __( 'License Active', 'learndash-convertkit' ),
					'license_inactive_label'   => __( 'License Inactive', 'learndash-convertkit' ),
					'save_activate_button'     => __( 'Save and Activate', 'learndash-convertkit' ),
				),
				'license_nag'            => array(
					'register_message' => _x( 'Register your copy of %s now to receive automatic updates and support.', '%s is the Plugin Name', 'learndash-convertkit' ),
					'purchase_message' => _x( 'If you do not have a license key, you can %1$spurchase one%2$s.', 'Both %s are used to place HTML for the <a> in the message', 'learndash-convertkit' ),
				),
				'license_activation'     => _x( '%s license successfully activated.', '%s is the Plugin Name', 'learndash-convertkit' ),
				'license_deletion'       => _x( '%s license successfully deleted.', '%s is the Plugin Name', 'learndash-convertkit' ),
				'license_deactivation'   => array(
					'error'   => _x( 'Error: could not deactivate the license for %s', '%s is the Plugin Name', 'learndash-convertkit' ),
					'success' => _x( '%s license successfully deactivated.', '%s is the Plugin Name', 'learndash-convertkit' ),
				),
				'license_error_messages' => array(
					'expired'             => _x( 'Your license key expired on %s.', '%s is a localized timestamp', 'learndash-convertkit' ),
					'revoked'             => __( 'Your license key has been disabled.', 'learndash-convertkit' ),
					'missing'             => __( 'Invalid license.', 'learndash-convertkit' ),
					'site_inactive'       => __( 'Your license is not active for this URL.', 'learndash-convertkit' ),
					'item_name_mismatch'  => _x( 'This appears to be an invalid license key for %s.', '%s is the Plugin Name', 'learndash-convertkit' ),
					'no_activations_left' => __( 'Your license key has reached its activation limit.', 'learndash-convertkit' ),
					'default'             => __( 'An error occurred, please try again.', 'learndash-convertkit' ),
				),
				'beta_checkbox'          => array(
					'label'      => __( 'Enable Beta Releases', 'learndash-convertkit' ),
					'disclaimer' => __( 'Beta Releases should not be considered as Stable. Enabling this on your Production Site is done at your own risk.', 'learndash-convertkit' ),
				),
			) );
		}

		/**
		 * Initializes the ConvertKit API.
		 *
		 * @since 1.0.0
		 * @access private
		 */
		private function setup_api() {

			require_once 'core/includes/class-ld-convertkit-api.php';

			$api_key    = get_option( 'ld_convertkit_api_key' );
			$api_secret = get_option( 'ld_convertkit_api_secret' );

			if ( $api_key && $api_secret ) {

				$this->convertkit_api = new LD_ConvertKit_API( $api_key, $api_secret );
			}
		}

		/**
		 * Registers plugin assets.
		 *
		 * @since 1.0.0
		 * @access private
		 */
		function register_assets() {

			wp_register_style(
				'learndash-convertkit-admin',
				LEARNDASH_CONVERTKIT_URL . '/assets/dist/css/learndash-convertkit-admin.min.css',
				array(),
				defined( 'WP_DEBUG' ) && WP_DEBUG ? time() : LEARNDASH_CONVERTKIT_VER
			);

			wp_register_script(
				'learndash-convertkit-admin',
				LEARNDASH_CONVERTKIT_URL . '/assets/dist/js/learndash-convertkit-admin.min.js',
				array(),
				defined( 'WP_DEBUG' ) && WP_DEBUG ? time() : LEARNDASH_CONVERTKIT_VER,
				true
			);

			wp_localize_script( 'learndash-convertkit-admin', 'LD_ConvertKit_Data', array(
				'l10n' => array(
					'could_not_get_tag_info' => __( 'Could not get Tag data. Please try again later.', 'learndash-convertkit' ),
					'view_tag'               => __( 'View Tag', 'learndash-convertkit' ),
				),
			) );
		}
	}

	// Load the bootstrapper
	require_once 'learndash-convertkit-bootstrapper.php';
	new LearnDash_ConvertKit_BootStrapper();

	/**
	 * Gets/loads the main plugin class.
	 *
	 * @since 1.0.0
	 *
	 * @return LearnDash_ConvertKit
	 */
	function LearnDash_ConvertKit() {

		return LearnDash_ConvertKit::instance();
	}
}