<?php

namespace uncanny_learndash_groups;


/**
 * Class WoocommerceBulkDiscount
 * @package uncanny_learndash_groups
 */
class WoocommerceBulkDiscount {
	/**
	 * WoocommerceBulkDiscount constructor.
	 */
	var $bulk_discount_calculated = false;
	var $discount_coeffs = [];

	/**
	 * WoocommerceBulkDiscount constructor.
	 */
	public function __construct() {

		// Only Run if woocommerce is available
		if ( Utilities::if_woocommerce_active() ) {
			add_action( 'wp_loaded', array( $this, 'save_bulk_discount' ), 111 );
			add_action( 'admin_menu', array( $this, 'create_bulk_discount_menu' ), 20 );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
			//add_action( 'woocommerce_cart_calculate_fees', array( $this, 'calculate_cart_discounts' ), 28, 1 );
			add_action( 'woocommerce_cart_calculate_fees', array( $this, 'calculate_cart_discounts_v2' ), 11, 1 );
		}
	}


	/**
	 *
	 */
	function create_bulk_discount_menu() {
		$capability   = 'manage_options';
		$menu_slug    = 'uncanny-groups-create-group';
		$submenu_slug = 'uncanny-learndash-groups-bulk-discount';
		add_submenu_page( $menu_slug, __( 'Bulk Discount', 'uncanny-learndash-groups' ), __( 'Bulk Discount', 'uncanny-learndash-groups' ), $capability, $submenu_slug, array(
			$this,
			'bulk_discount_func',
		) );

	}

	/**
	 *
	 */
	function bulk_discount_func() {
		include Utilities::get_template( 'admin-bulk-discounts.php' );

	}

	/**
	 * @param $hook
	 */
	public function admin_scripts( $hook ) {
		if ( strpos( $hook, 'uncanny-learndash-groups-bulk-discount' ) ) {
			wp_enqueue_script( 'ulgm-backend', Utilities::get_asset( 'backend', 'bundle.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_style( 'ulgm-backend', Utilities::get_asset( 'backend', 'bundle.min.css' ), array(), Utilities::get_version() );
		}
	}

	/**
	 *
	 */
	function save_bulk_discount() {
		if ( isset( $_POST['_ulgm_bulk_nonce'] ) && wp_verify_nonce( $_POST['_ulgm_bulk_nonce'], Utilities::get_plugin_name() ) ) {
			if ( isset( $_POST['action'] ) && 'save-bulk-discount' === $_POST['action'] ) {

				$bulk_discount = array();
				if ( isset( $_POST['ulgm-enable-bulk-discount'] ) && 'on' === $_POST['ulgm-enable-bulk-discount'] ) {
					$bulk_discount['enabled'] = 'yes';
				} else {
					$bulk_discount['enabled'] = 'no';
				}

				for ( $i = 1; $i <= 10; $i ++ ) {
					if ( isset( $_POST[ 'ulgm_bulk_discount_quantity_' . $i ] ) && ! empty( $_POST[ 'ulgm_bulk_discount_quantity_' . $i ] ) ) {
						$bulk_discount['discounts'][ $i ]['qty'] = $_POST[ 'ulgm_bulk_discount_quantity_' . $i ];
					}
					if ( isset( $_POST[ '_ulgm_bulk_discount_value_' . $i ] ) && ! empty( $_POST[ '_ulgm_bulk_discount_value_' . $i ] ) ) {
						$bulk_discount['discounts'][ $i ]['percent'] = $_POST[ '_ulgm_bulk_discount_value_' . $i ];
					}
				}
				update_option( SharedFunctions::$bulk_discount_options, $bulk_discount );
			}
			wp_safe_redirect( $_POST['_wp_http_referer'] );
			exit;
		}
	}

	/**
	 * Gather discount information to the array $this->discount_coefs
	 */
	protected function gather_discount_coeffs() {
		//
		$quantities = [];
		$quantity   = WC()->cart->get_cart_item_quantities();

		////
		$user_id       = wp_get_current_user()->ID;
		$get_transient = get_transient( '_ulgm_user_' . $user_id . '_order' );
		if ( is_user_logged_in() && ! empty( $get_transient ) ) {
			$product = wc_get_product( absint( $get_transient['order_details']['product_id'] ) );
			if ( $product->is_type( 'license' ) ) {
				$quantities[ $product->get_id() ][] = absint( $get_transient['existing_qty'] );
			}
		}

		if ( $quantity ) {
			foreach ( $quantity as $product_id => $qty ) {
				$product = wc_get_product( $product_id );
				if ( $product->is_type( 'license' ) ) {
					$quantities[ $product_id ][] = $qty;
				}
			}
		}
		Utilities::log( [ $quantities, $quantity ], '[ $quantities, $quantity ]', true, 'newissue' );
		foreach ( $quantities as $product_id => $qty ) {
			if ( 0 !== absint( $product_id ) && ! key_exists( $product_id, $this->discount_coeffs ) ) {
				$total_qty                                          = absint( array_sum( $quantities[ $product_id ] ) );
				$this->discount_coeffs[ $product_id ]['coeff']      = $this->get_discounted_coeff( $total_qty );
				$this->discount_coeffs[ $product_id ]['orig_price'] = wc_get_price_excluding_tax( wc_get_product( $product_id ) );
				$this->discount_coeffs[ $product_id ]['quantity']   = $total_qty;
			}
		}

	}


	/**
	 * For given product, and quantity return the price modifying factor (percentage discount) or value to deduct (flat & fixed discounts).
	 *
	 * @param $product_id
	 * @param $quantity
	 *
	 * @return float
	 */
	protected function get_discounted_coeff( $quantity ) {

		$q = array( 0.0 );
		$d = array( 0.0 );

		/* Find the appropriate discount coefficient by looping through up to the three discount settings */
		$bulk_discounts = get_option( SharedFunctions::$bulk_discount_options, array() );
		if ( ! empty( $bulk_discounts ) ) {
			if ( 'yes' === $bulk_discounts['enabled'] ) {
				$available_discounts = $bulk_discounts['discounts'];
				for ( $i = 1; $i <= 10; $i ++ ) {
					if ( key_exists( $i, $available_discounts ) ) {
						array_push( $q, $available_discounts[ $i ]['qty'] );
						array_push( $d, $available_discounts[ $i ]['percent'] ? $available_discounts[ $i ]['percent'] : 0.0 );

						if ( $quantity >= $q[ $i ] && $q[ $i ] > $q[0] ) {
							$q[0] = $q[ $i ];
							$d[0] = $d[ $i ];
						}
					}
				}

				return min( 1.0, max( 0, ( 100.0 - round( $d[0], 2 ) ) / 100.0 ) );
			} else {
				return 1.0;
			}
		} else {
			return 1.0;
		}
	}

	/**
	 *
	 * @param \WC_Cart $cart
	 */
	public function calculate_cart_discounts_v2( \WC_Cart $cart ) {
		$this->gather_discount_coeffs();
		$bulk_discounts = get_option( SharedFunctions::$bulk_discount_options, array() );
		$coeff          = 1;
		if ( ! empty( $bulk_discounts ) && 'yes' === $bulk_discounts['enabled'] ) {
			if ( count( $cart->cart_contents ) > 0 ) {
				foreach ( $cart->cart_contents as $cart_item_key => $values ) {
					$apply_coeff  = 0;
					$_product     = $values['data'];
					$_price       = wc_get_price_excluding_tax( $_product );
					$_quantity    = 0;
					$subtract_fee = 0;
					if ( key_exists( absint( $_product->get_id() ), $this->discount_coeffs ) ) {
						$coeff         = $this->discount_coeffs[ absint( $_product->get_id() ) ]['coeff'];
						$previous_fees = \WC()->cart->get_fees();
						//$previous_fees = floatval($previous_fees) + floatval(\WC()->cart->get_discount_total());
						if ( $previous_fees ) {
							foreach ( $previous_fees as $fee ) {
								if ( ! strpos( $fee->name, 'License Course Deduction' ) ) {
									$subtract_fee += $fee->amount;
								}
							}
						}
					}
					$subtract_fee = floatval( $subtract_fee ) - \WC()->cart->get_discount_total();

					if ( $_product->is_type( 'license' ) ) {
						$_quantity   = $values['quantity'];
						$_price      = floatval( $_price ) * absint( $_quantity );
						$apply_coeff = 1;
					} else {
						$apply_coeff = 0;
					}
					if ( 1 == $coeff ) {
						return;
					} else {
						if ( 1 === $apply_coeff ) {
							$discount = ( $_price + $subtract_fee ) * ( 1 - $coeff );
							\WC()->cart->add_fee( sprintf( __( '%s%% Bulk Discount', 'uncanny-learndash-groups' ), ( 1 - $coeff ) * 100 ), - $discount );
						}
					}
				}

				$this->bulk_discount_calculated = true;
			}
		}
	}

	/**
	 * @param $_product
	 *
	 * @return mixed
	 */
	protected function get_product_id( $_product ) {
		if ( version_compare( WOOCOMMERCE_VERSION, '2.7.0' ) >= 0 ) {
			return $_product->get_id();
		}

		return $_product->id;
	}

	/**
	 * @param $price
	 *
	 * @return string
	 */
	protected function get_price( $price ) {
		if ( version_compare( WOOCOMMERCE_VERSION, '2.7.0' ) >= 0 ) {
			return wc_price( $price );
		} else {
			return woocommerce_price( $price );
		}
	}

	/**
	 * @param $id
	 *
	 * @return \WC_Product
	 */
	protected function get_product( $id ) {
		if ( version_compare( WOOCOMMERCE_VERSION, '2.7.0' ) >= 0 ) {
			return wc_get_product( $id );
		} else {
			return get_product( $id );
		}
	}

	/**
	 * @param $product
	 *
	 * @return int
	 */
	protected function get_actual_id( $product ) {

		if ( $product instanceof \WC_Product_Variation ) {
			return $product->variation_id;
		} else {
			return $this->get_product_id( $product );
		}

	}

}