<?php

namespace Uncanny_Automator_Pro;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * This class autoloads all files within specified directories
 * and runs EDD plugin licensing and updater
 *
 * Class Boot
 * @package Uncanny_Automator_Pro
 */
class Boot {

	/**
	 * The instance of the class
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Boot
	 */
	private static $instance = null;

	/**
	 * The parent plugins namespace
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array
	 */
	private static $parent_namespace = null;

	/**
	 * The directories that are auto loaded and initialized
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array
	 */
	private static $auto_loaded_directories = null;

	/**
	 * class constructor
	 */
	private function __construct() {

		// We need to check if spl auto loading is available when activating plugin
		// Plugin will not activate if SPL extension is not enabled by throwing error
		if ( ! extension_loaded( 'SPL' ) ) {
			trigger_error( esc_html__( 'Please contact your hosting company to update to php version 5.3+ and enable spl extensions.', 'uncanny-owl-customizations-well-drilling' ), E_USER_ERROR );
		}

		add_action( 'plugins_loaded', array( $this, 'boot_child_plugin' ), 9 );

		spl_autoload_register( array( $this, 'require_class_files' ) );

	}

	public function boot_child_plugin() {

		if ( ! defined( 'UAP_AUTOMATOR_FILE_' ) ) {
			add_action( 'admin_notices', array( $this, 'lite_needs_to_be_installed' ) );

			return;
		}

		include_once( Utilities::get_include( 'internal-triggers-actions.php' ) );
		$this->internal_triggers_actions = new Internal_Triggers_Actions();

		// Initialize all classes in given directories
		$this->auto_initialize_classes();

		/* Licensing */

		// URL of store powering the plugin
		define( 'UAP_AUTOMATOR_PRO_STORE_URL', 'https://www.automatorplugin.com/' ); // you should use your own CONSTANT name, and be sure to replace it throughout this file

		// Store download name/title
		define( 'UAP_AUTOMATOR_PRO_ITEM_NAME', 'Uncanny Automator Pro' ); // you should use your own CONSTANT name, and be sure to replace it throughout this file

		// include updater
		$updater = Utilities::get_include( 'EDD_SL_Plugin_Updater.php' );
		include_once( $updater );

		add_action( 'admin_init', array( $this, 'uap_automator_plugin_updater' ), 0 );
		add_action( 'admin_menu', array( $this, 'uap_automator_license_menu' ), 11 );
		add_action( 'admin_init', array( $this, 'uap_automator_register_option' ) );
		add_action( 'admin_init', array( $this, 'uap_automator_activate_license' ) );
		add_action( 'admin_init', array( $this, 'uap_automator_deactivate_license' ) );
	}

	/**
	 * Run error notice in Uncanny Automator Lite is not installed
	 */
	public function lite_needs_to_be_installed() {

		$class   = 'notice notice-error';

		if( defined( 'UAP_AUTOMATOR_FILE' ) ){
			// An old version of Automator Lite is running
			$message = __( 'Uncanny Automator Lite needs to be updated before it can be used with the new updates and enhancements of Automator Pro.', 'uncanny-automator-pro' );
		}else{
			$message = __( 'Please activate Uncanny Automator Lite before activating Uncanny Automator Pro.', 'uncanny-automator-pro' );
		}

		printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), esc_html( $message ) );
	}

	/**
	 * Run the EDD updater class
	 */
	public function uap_automator_plugin_updater() {

		// retrieve our license key from the DB
		$license_key = trim( get_option( 'uap_automator_pro_license_key' ) );

		// setup the updater
		new EDD_SL_Plugin_Updater( UAP_AUTOMATOR_PRO_STORE_URL, UAP_AUTOMATOR_PRO_FILE, array(
			'version'   => Utilities::get_version(),                // current version number
			'license'   => $license_key,        // license key (used get_option above to retrieve from DB)
			'item_name' => UAP_AUTOMATOR_PRO_ITEM_NAME,    // name of this plugin
			'author'    => 'Uncanny Owl'  // author of this plugin
		) );

	}

	/**
	 * Licence options page
	 */
	public function uap_automator_license_menu() {
		add_submenu_page( 'uncanny-activities', 'Uncanny Automator License Activation', 'License Activation', 'manage_options', 'uncanny-automator-license-activation', array(
			$this,
			'uap_automator_license_page'
		) );
	}

	/**
	 * Licensing page html output
	 */
	public function uap_automator_license_page() {

		// retrieve the license from the database
		$license = trim( get_option( 'uap_automator_pro_license_key' ) );


		// data to send in our API request
		$api_params = array(
			'edd_action' => 'check_license',
			'license'    => $license,
			'item_name'  => urlencode( UAP_AUTOMATOR_PRO_ITEM_NAME ), // the name of our product in uo
			'url'        => home_url()
		);

		// Call the custom API.
		$response = wp_remote_post( UAP_AUTOMATOR_PRO_STORE_URL, array(
			'timeout'   => 15,
			'sslverify' => false,
			'body'      => $api_params
		) );

		// make sure the response came back okay
		if ( is_wp_error( $response ) ) {
			return false;
		}

		// decode the license data
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( 'valid' !== $license_data->license ) {
			// $license_data->license will be either "valid", "invalid", "expired", "disabled", or "inactive"
			update_option( 'uap_automator_pro_license_status', $license_data->license );
		}


		// $license_data->license_check will be either "valid", "invalid", "expired", "disabled", or "inactive"
		update_option( 'uap_automator_pro_license_check', $license_data->license );


		include( Utilities::get_view( 'admin-licensing-settings-page.php' ) );
	}

	/**
	 * Licensing page styles
	 *
	 * @param $hook
	 */
	public function scripts( $hook ) {

		if ( strpos( $hook, 'uncanny-automator-license-activation' ) ) {

			wp_enqueue_style( 'uap-admin-license', Utilities::get_css( 'admin/license.css' ), array(), Utilities::get_version() );

		}

	}

	/**
	 * Register settings
	 */
	public function uap_automator_register_option() {
		// creates our settings in the options table
		register_setting( 'uap_automator_pro_license', 'uap_automator_pro_license_key', array(
			__CLASS__,
			'uap_automator_sanitize_license'
		) );
	}

	/**
	 * License Sanitation
	 */
	public function uap_automator_sanitize_license( $new ) {
		$old = get_option( 'uap_automator_pro_license_key' );
		if ( $old && $old != $new ) {
			delete_option( 'uap_automator_pro_license_status' ); // new license has been entered, so must reactivate
		}

		return $new;
	}

	/**
	 * License Activation
	 *
	 * @return bool|void
	 */
	public function uap_automator_activate_license() {

		// listen for our activate button to be clicked
		if ( isset( $_POST['uap_automator_pro_license_activate'] ) ) {

			// run a quick security check
			if ( ! check_admin_referer( 'uap_automator_pro_nonce', 'uap_automator_pro_nonce' ) ) {
				return;
			} // get out if we didn't click the Activate button

			// retrieve the license from the database
			$license = trim( get_option( 'uap_automator_pro_license_key' ) );


			// data to send in our API request
			$api_params = array(
				'edd_action' => 'activate_license',
				'license'    => $license,
				'item_name'  => urlencode( UAP_AUTOMATOR_PRO_ITEM_NAME ), // the name of our product in uo
				'url'        => home_url()
			);

			// Call the custom API.
			$response = wp_remote_post( UAP_AUTOMATOR_PRO_STORE_URL, array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params
			) );

			// make sure the response came back okay
			if ( is_wp_error( $response ) ) {
				return false;
			}

			// decode the license data
			$license_data = json_decode( wp_remote_retrieve_body( $response ) );

			// $license_data->license will be either "valid" or "invalid"
			update_option( 'uap_automator_pro_license_status', $license_data->license_check );

			// $license_data->license_check will be either "valid", "invalid", "expired", "disabled", "inactive", or "site_inactive"
			update_option( 'uap_automator_pro_license_status', $license_data->license );

		}
	}

	/**
	 * License deactivation
	 *
	 * @return bool|void
	 */
	public function uap_automator_deactivate_license() {

		// listen for our activate button to be clicked
		if ( isset( $_POST['uap_automator_pro_license_deactivate'] ) ) {

			// run a quick security check
			if ( ! check_admin_referer( 'uap_automator_pro_nonce', 'uap_automator_pro_nonce' ) ) {
				return;
			} // get out if we didn't click the Activate button

			// retrieve the license from the database
			$license = trim( get_option( 'uap_automator_pro_license_key' ) );


			// data to send in our API request
			$api_params = array(
				'edd_action' => 'deactivate_license',
				'license'    => $license,
				'item_name'  => urlencode( UAP_AUTOMATOR_PRO_ITEM_NAME ), // the name of our product in uo
				'url'        => home_url()
			);

			// Call the custom API.
			$response = wp_remote_post( UAP_AUTOMATOR_PRO_STORE_URL, array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params
			) );

			// make sure the response came back okay
			if ( is_wp_error( $response ) ) {
				return false;
			}

			// decode the license data
			$license_data = json_decode( wp_remote_retrieve_body( $response ) );

			// $license_data->license will be either "deactivated" or "failed"
			if ( $license_data->license == 'deactivated' || $license_data->license == 'failed' ) {
				delete_option( 'uap_automator_pro_license_status' );
			}

		}
	}


	/**
	 * Creates singleton instance of Boot class and defines which directories are auto loaded
	 *
	 * @since 1.0.0
	 *
	 * @param string $parent_namespace
	 * @param array $auto_loaded_directories
	 *
	 * @return Boot
	 */
	public static function get_instance(
		$parent_namespace = false, $auto_loaded_directories = [
		'core/classes/',
		'core/extensions/'
	]
	) {

		if ( null === self::$instance ) {

			// Define directories were the auto loader looks for files and initializes them
			self::$auto_loaded_directories = $auto_loaded_directories;

			// Define if it exists, the parent plugins namespace
			self::$parent_namespace = $parent_namespace;

			// Lets boot up!
			self::$instance = new self();
		}

		return self::$instance;
	}


	/**
	 * SPL Auto Loader functions
	 *
	 * @since 1.0.0
	 *
	 * @param string $class Any
	 */
	private function require_class_files( $class ) {

		// Remove Class's namespace eg: my_namespace/MyClassName to MyClassName
		$class = str_replace( __NAMESPACE__, '', $class );
		$class = str_replace( '\\', '', $class );

		// Replace _ with - eg. eg: My_Class_Name to My-Class-Name
		$class_to_filename = str_replace( '_', '-', $class );

		// Create file name that will be loaded from the classes directory eg: My-Class-Name to my-class-name.php
		$file_name = strtolower( $class_to_filename ) . '.php';

		// Check each directory
		foreach ( self::$auto_loaded_directories as $directory ) {

			$file_path = dirname( __FILE__ ) . DIRECTORY_SEPARATOR . $directory . DIRECTORY_SEPARATOR . $file_name;

			// Does the file exist
			if ( file_exists( $file_path ) ) {

				// File found, require it
				require_once( $file_path );

				// You can cannot have duplicate files names. Once the first file is found, the loop ends.
				return;
			}
		}

	}

	/**
	 * Looks through all defined directories and modifies file name to create new class instance.
	 *
	 * @since 1.0.0
	 *
	 */
	private function auto_initialize_classes() {

		// Check each directory
		foreach ( self::$auto_loaded_directories as $directory ) {

			// Get all files in directory
			$files = scandir( dirname( __FILE__ ) . DIRECTORY_SEPARATOR . $directory );

			// remove parent directory, sub directory, and silence is golden index.php
			$files = array_diff( $files, array( '..', '.', 'index.php' ) );

			// Loop through all files in directory to create class names from file name
			foreach ( $files as $file ) {

				// Remove file extension my-class-name.php to my-class-name
				$file_name = str_replace( '.php', '', $file );

				// Split file name on - eg: my-class-name to array( 'my', 'class', 'name')
				$class_to_filename    = explode( '-', $file_name );
				$plugin_from_filename = $class_to_filename[0];


				// Make the first letter of each word in array upper case - eg array( 'my', 'class', 'name') to array( 'My', 'Class', 'Name')
				$class_to_filename = array_map( function ( $word ) {
					return ucfirst( $word );
				}, $class_to_filename );

				// Implode array into class name - eg. array( 'My', 'Class', 'Name') to MyClassName
				$class_name = implode( '_', $class_to_filename );

				$class = __NAMESPACE__ . '\\' . $class_name;

				// We way want to include some class with the autoloader but not initialize them ex. interface class
				$skip_classes = apply_filters( 'Skip_class_initialization', array(), $directory, $files, $class, $class_name );
				if ( in_array( $class_name, $skip_classes ) ) {
					continue;
				}
				if ( 'yes' === $this->required_plugin_active( $plugin_from_filename ) ) {

					if ( self::$parent_namespace ) {

						$parent_plugin_utilities = '\\' . self::$parent_namespace . '\\' . 'Utilities';

						if ( class_exists( $parent_plugin_utilities ) ) {
							call_user_func( array(
								$parent_plugin_utilities,
								'add_class_instance'
							), $class, new $class );
						} else {
							// Warning that a parent plugin is set but the utilities class or the namespace was not found
							trigger_error( "Parent plugin is set. The namespace maybe incorrect or the utilities class hasn't loaded.", E_USER_WARNING );
						}

					} else {
						Utilities::add_class_instance( $class, new $class );
					}
				}
			}
		}
	}

	/**
	 * @param $plugin_to_check
	 *
	 * @return bool
	 */
	public function required_plugin_active( $plugin_to_check ) {
		$plugin = trim( $plugin_to_check );
		$active = 'no';

		include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		switch ( $plugin ) {
			case 'wp';
				$active = 'yes';
				break;
			case 'gf':
				if ( is_plugin_active( 'gravityforms/gravityforms.php' ) ) {
					$active = 'yes';
				}
				break;
			case 'ld':
				if ( is_plugin_active( 'sfwd-lms/sfwd_lms.php' ) ) {
					$active = 'yes';
				}
				break;
			case 'wc':
				if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
					$active = 'yes';
				} elseif ( is_plugin_active_for_network( 'woocommerce/woocommerce.php' ) ) {
					$active = 'yes';
				} else {
					$active = 'no';
				}
				break;
			default:
				$active = 'yes';
				break;
		}

		return $active;
	}

	/**
	 * Make clone magic method private, so nobody can clone instance.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {
	}

	/**
	 * Make sleep magic method private, so nobody can serialize instance.
	 *
	 * @since 1.0.0
	 */
	private function __sleep() {
	}

	/**
	 * Make wakeup magic method private, so nobody can unserialize instance.
	 *
	 * @since 1.0.0
	 */
	private function __wakeup() {

	}

}





