<?php

namespace Uncanny_Automator_Pro;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * This class is used to run any configurations before the plugin is initialized
 *
 * @package    Private_Plugin_Boilerplate
 */
class Config {


	/**
	 * The instance of the class
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Boot
	 */
	private static $instance = null;

	/**
	 * Creates singleton instance of class
	 *
	 * @return Config $instance The UncannyLearnDashGroupManagement Class
	 * @since 1.0.0
	 *
	 */
	public static function get_instance() {

		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Initialize the class and setup its properties.
	 *
	 * @param string $plugin_name The name of the plugin
	 * @param string $prefix The variable used to prefix filters and actions
	 * @param string $version The version of this plugin
	 * @param string $file The main plugin file __FILE__
	 * @param bool $debug Whether debug log in php and js files are enabled
	 *
	 * @since    1.0.0
	 *
	 */
	public function configure_plugin_before_boot( $plugin_name, $prefix, $version, $file, $debug ) {


		$this->define_constants( $plugin_name, $prefix, $version, $file, $debug );

		do_action( Utilities::get_prefix() . '_define_constants_after' );

		register_activation_hook( Utilities::get_plugin_file(), array( $this, 'activation' ) );

		register_deactivation_hook( Utilities::get_plugin_file(), array( $this, 'deactivation' ) );

		do_action( Utilities::get_prefix() . '_config_setup_after' );

		add_action( 'upgrader_process_complete', array( $this, 'plugin_updated' ), 10, 2 );
	}

	/**
	 *
	 * This action is documented in includes/class-plugin-name-deactivator.php
	 *
	 * @param string $plugin_name The name of the plugin
	 * @param string $prefix Variable used to prefix filters and actions
	 * @param string $version The version of this plugin.
	 * @param string $plugin_file The main plugin file __FILE__
	 * @param string $debug_mode Whether debug log in php and js files are enabled
	 *
	 * @since    1.0.0
	 * @access   private
	 *
	 */
	private function define_constants( $plugin_name, $prefix, $version, $plugin_file, $debug_mode ) {


		// Set and define version
		if ( ! defined( strtoupper( $prefix ) . '_PLUGIN_NAME' ) ) {
			define( strtoupper( $prefix ) . '_PLUGIN_NAME', $plugin_name );
			Utilities::set_plugin_name( $plugin_name );
		}

		// Set and define version
		if ( ! defined( strtoupper( $prefix ) . '_VERSION' ) ) {
			define( strtoupper( $prefix ) . '_VERSION', $version );
			Utilities::set_version( $version );
		}

		// Set and define prefix
		if ( ! defined( strtoupper( $prefix ) . '_PREFIX' ) ) {
			define( strtoupper( $prefix ) . '_PREFIX', $prefix );
			Utilities::set_prefix( $prefix );
		}

		// Set and define the main plugin file path
		if ( ! defined( $prefix . '_FILE' ) ) {
			define( strtoupper( $prefix ) . '_FILE', $plugin_file );
			Utilities::set_plugin_file( $plugin_file );
		}

		// Set and define debug mode
		if ( ! defined( $prefix . '_DEBUG_MODE' ) ) {
			define( strtoupper( $prefix ) . '_DEBUG_MODE', $debug_mode );
			Utilities::set_debug_mode( $debug_mode );
		}

		// Set and define the server initialization ( Server time and not to be confused with WP current_time() )
		if ( ! defined( $prefix . '_SERVER_INITIALIZATION' ) ) {
			$time = time();
			define( strtoupper( $prefix ) . '_SERVER_INITIALIZATION', $time );
			Utilities::set_plugin_initialization( $time );
		}

		Utilities::log(
			array(
				'get_plugin_name'           => Utilities::get_plugin_name(),
				'get_version'               => Utilities::get_version(),
				'get_prefix'                => Utilities::get_prefix(),
				'get_plugin_file'           => Utilities::get_plugin_file(),
				'get_debug_mode'            => Utilities::get_debug_mode(),
				'get_plugin_initialization' => date( Utilities::get_date_time_format(), Utilities::get_plugin_initialization() )

			),
			'Configuration Variables'
		);

	}


	/**
	 * The code that runs during plugin activation.
	 * @since    1.0.0
	 */
	public function activation() {

		do_action( Utilities::get_prefix() . 'uapro_activation_before' );

		if ( ! function_exists( 'as_schedule_recurring_action' ) ) {
			wp_schedule_event( time(), 'daily', 'uapro_auto_purge_logs' );
		} else {
			as_schedule_recurring_action( strtotime( 'midnight tonight' ), DAY_IN_SECONDS, 'uapro_auto_purge_logs' );
		}

		do_action( Utilities::get_prefix() . '_activation_after' );

	}

	/**
	 * The code that runs after plugin update.
	 *
	 * @param \WP_Upgrader $upgrader_object
	 * @param $hook_extra
	 *
	 * @since    2.0.0
	 */
	public function plugin_updated( \WP_Upgrader $upgrader_object, $hook_extra ) {
		if ( ( key_exists( 'action', $hook_extra ) && 'update' === $hook_extra['action'] ) && ( key_exists( 'type', $hook_extra ) && 'plugin' === $hook_extra['type'] ) && key_exists( 'plugins', $hook_extra ) && is_array( $hook_extra['plugins'] ) && ! empty( $hook_extra['plugins'] ) ) {
			// if this action is update plugin.
			$this_plugin = UAP_AUTOMATOR_PRO_FILE;
			foreach ( $hook_extra['plugins'] as $each_plugin ) {
				if ( $each_plugin === $this_plugin ) {
					if ( wp_get_scheduled_event( 'uapro_auto_purge_logs' ) ) {
						//Remove already scheduled cron
						wp_clear_scheduled_hook( 'uapro_auto_purge_logs' );
						if ( ! function_exists( 'as_schedule_recurring_action' ) ) {
							wp_schedule_event( time(), 'daily', 'uapro_auto_purge_logs' );
						} else {
							//Add Action Scheduler event
							as_schedule_recurring_action( strtotime( 'midnight tonight' ), DAY_IN_SECONDS, 'uapro_auto_purge_logs' );
						}
					}
				}
			}
		}
	}


	/**
	 * The code that runs during plugin deactivation.
	 * @since    1.0.0
	 */
	public function deactivation() {

		do_action( Utilities::get_prefix() . '_deactivation_before' );

		wp_clear_scheduled_hook( 'uapro_auto_purge_logs' );

		do_action( Utilities::get_prefix() . '_deactivation_after' );

	}
}